# coding=utf-8

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
from clcommon import FormattedException

# Possible interpreter statuses
ENABLED_STATUS = 'enabled'
DISABLED_STATUS = 'disabled'
NOT_INSTALLED_STATUS = 'not_installed'
INSTALLING_STATUS = 'installing'
REMOVING_STATUS = 'removing'

# Means that cache update process is currently running
CACHE_UPDATING_YUM_STATUS = "cache_updating"

# possible application states
APP_STARTED_CONST = 'started'
APP_STOPPED_CONST = 'stopped'


class BaseSelectorError(FormattedException):
    """Top level error class for admin's part of selector"""
    pass


class AcquireInterpreterLockError(BaseSelectorError):
    """Error raised when we are unable to lock interpreter version"""
    def __init__(self, version):
        super(AcquireInterpreterLockError, self).__init__({
            "message":
                "Unable to lock specified interpreter version %(version)s. Probably it's not "
                "installed at this moment",
            "context": {
                'version': version
            },
        })


class AcquireApplicationLockError(BaseSelectorError):
    """Error raised when we are unable to lock application"""
    def __init__(self, app_root, reason=None):
        message = "Can't acquire lock for app: %(app)s"
        if reason:
            message += " Reason is: {}".format(reason)
        super(AcquireApplicationLockError, self).__init__({
            'message': message,
            'context': {
                'app': app_root,
            },
        })


class MissingVirtualenvError(BaseSelectorError):
    """Error raised when virtualenv directory is absent"""
    def __init__(self, app_venv):
        super(MissingVirtualenvError, self).__init__({
            'message':
                "No such application or it's broken. "
                "Unable to find app venv folder by this path: '%(app_venv)s'",
            'context': {
                'app_venv': app_venv,
            },
        })


class MissingAppRootError(BaseSelectorError):
    """Error raised when application root directory is absent"""
    def __init__(self, app_root):
        super(MissingAppRootError, self).__init__({
            'message':
                "No such application or it's broken. "
                "Unable to find app-root folder by this path: '%(app_root)s'",
            'context': {
                'app_root': app_root,
            },
        })


class AbsentFileError(BaseSelectorError):
    """Error raised when some file does not exist"""
    def __init__(self, filepath):
        super(AbsentFileError, self).__init__(
            {'message': "File %(file)s is absent.",
             'context': {'file': filepath}}
        )


class UnsupportedSelectorError(BaseSelectorError):
    def __init__(self, interpreter):
        super(UnsupportedSelectorError, self).__init__(
            {'message': 'Unsupported interpreter was passed: "%(interpreter)s"',
             'context': {'interpreter': interpreter}}
        )
