# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import glob
import os

from clselect.clselectexcept import ClSelectExcept
from clselect.utils import check_output, demote


ALT_PATH = '/opt/alt'
INT_BASE = 'ruby'
INT_VERS = '??'


class Interpreter(object):

    def __init__(self, prefix=None, binary=None, version=None,
                 version_full=None, target_user=None):
        self.user = target_user
        if prefix and binary:
            self.prefix = prefix
            self.binary = binary
        elif prefix:
            self.prefix = prefix
            ruby_bin = os.path.join(self.prefix, 'bin', 'ruby')
            if (os.path.isfile(ruby_bin) and not os.path.islink(ruby_bin)):
                binary = ruby_bin

            if not binary:
                raise ClSelectExcept.InterpreterError(
                    'Can not find ruby binary')

            self.binary = binary
        elif binary:
            self.binary = binary
            self.prefix = os.path.abspath(os.path.join(
                os.path.dirname(binary), os.path.pardir))
        else:
            interpreter = interpreters('version')['2.1']
            self.binary = interpreter.binary
            self.prefix = interpreter.prefix
            self.version = interpreter.version
            self.version_full = interpreter.version_full

        if version:
            self.version = version
        else:
            self.version = None

        if version_full:
            self.version_full = version_full
        else:
            try:
                if self.user:
                    version_raw = check_output(self.binary, '-e', 'puts RUBY_VERSION',
                                               preexec_fn=demote(self.user))
                else:
                    version_raw = check_output(self.binary, '-e', 'puts RUBY_VERSION')
            except (OSError, ClSelectExcept.ExternalProgramFailed):
                raise ClSelectExcept.InterpreterError(
                    'Error checking ruby binary version')
            self.version_full = version_raw.strip()

        if not self.version:
            self.version = '.'.join(self.version_full.split('.')[:2])

    def __repr__(self):
        return (
            "%s.%s(prefix='%s', version='%s', binary='%s', "
            "version_full='%s')" % (
                self.__class__.__module__, self.__class__.__name__,
                self.prefix, self.version, self.binary, self.version_full
            ))

    def as_dict(self, key=None):
        i = {
            'prefix': self.prefix,
            'binary': self.binary,
            'version': self.version,
            'version_full': self.version_full,
            }
        if key:
            del i[key]
            return {getattr(self, key): i}
        return i


def interpreters(key=None):
    search_path = os.path.join(ALT_PATH, INT_BASE + INT_VERS)
    interps = []
    for path in glob.glob(search_path):
        try:
            interpreter = Interpreter(path)
        except ClSelectExcept.InterpreterError:
            continue
        interps.append(interpreter)
    if key:
        return dict((getattr(i, key), i) for i in interps) 
    else:
        return interps


def interpreters_dict(key):
    return dict(list(i.as_dict(key).items())[0] for i in interpreters())
