# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import

import getopt
import os
import shutil
import sys

from clselect.utils import mkdir_p


WRAP_LIST = [
    ['node', []],
    ['npm', []],
]

SET_ENV_SCRIPT = 'set_env_vars.py'
ACTIVATE_SCRIPT = 'activate'
NODE_WRAPPER = 'node_wrapper'
NPM_WRAPPER = 'npm_wrapper'
NODE = 'node'
NPM = 'npm'
BIN_DIR = '/usr/share/l.v.e-manager/utils/'


def fail_with(msg, exit_code=1):
    print(msg, file=sys.stderr)
    exit(exit_code)


def create(destination, node=None, destroy_first=False):
    destination = os.path.expanduser(destination)

    venv_bin = os.path.join(destination, 'bin')

    if destroy_first:
        try:
            shutil.rmtree(os.path.join(destination, 'bin'))
        except (IOError, OSError):
            pass

    mkdir_p(venv_bin)

    set_env_script_abs = os.path.join(venv_bin, SET_ENV_SCRIPT)
    activate_abs = os.path.join(venv_bin, ACTIVATE_SCRIPT)
    node_wrapper_abs = os.path.join(venv_bin, NODE)
    npm_wrapper_abs = os.path.join(venv_bin, NPM)

    link_set_env_script = os.path.join(BIN_DIR, SET_ENV_SCRIPT)
    link_activate_path = os.path.join(BIN_DIR, ACTIVATE_SCRIPT)
    link_node_path = os.path.join(BIN_DIR, NODE_WRAPPER)
    link_npm_path = os.path.join(BIN_DIR, NPM_WRAPPER)

    os.symlink(link_activate_path, activate_abs)
    os.symlink(link_set_env_script, set_env_script_abs)
    os.symlink(link_node_path, node_wrapper_abs)
    os.symlink(link_npm_path, npm_wrapper_abs)


def main():
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'n:r', ['node=', 'recreate-bin'])
    except getopt.GetoptError as err:
        print(str(err))
        usage()
        sys.exit(1)

    node = None
    destroy_first = False

    if not args:
        print('no destination specified')
        usage()
        sys.exit(1)

    destination = args[0]

    for o, a in opts:
        if o in ('-n', '--node'):
            node = a
        elif o in ('-r', '--recreate-bin'):
            destroy_first = True
        else:
            print('unhandled option')
            sys.exit(1)

    try:
        create(destination, node, destroy_first=destroy_first)
    except OSError as e:
        fail_with(e.strerror)


def usage():
    print('Usage: ' + sys.argv[0] + ' [OPTIONS] VENV')
    print()
    print('Options:')
    print(' -n, --node PATH       Node interpreter to use to create new venv')
    print(' -h, --help            Show this message')


if __name__ == '__main__':
    main()
