# coding: utf-8

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import os

from clselect.baseclselect import DISABLED_STATUS, ENABLED_STATUS
from clselect.baseclselect.config import BaseSelectorConfig
from . import (
    CONFIG_DIR,
    create_config_dirs,
    ALT_PYTHON_PREFIX
)
from clconfig.ui_config_lib import get_ui_config


class PythonConfig(BaseSelectorConfig):
    """
    It's responsible for all interaction with CL Python selector config file
    and computing some effective values when config is absent or incomplete
    """

    _config_file = os.path.join(CONFIG_DIR, 'selector.json')

    def __init__(self, pkg_manager):
        """
        :param pkg_manager: instance of PkgManager class
        """
        super(PythonConfig, self).__init__(pkg_manager)

        self.reload()
        self.set_initial_selector_status()

    def _create_config_dirs(self):
        return create_config_dirs()

    @property
    def available_versions(self):
        """
        Returns a dict with status of all installed alt-python versions
        :rtype: dict
        """
        disabled = self.Cfg.disabled_versions or []
        return {
            ver: {
                'status': DISABLED_STATUS if ver in disabled else ENABLED_STATUS,
                'base_dir': ALT_PYTHON_PREFIX + ''.join(ver.split('.')[:2]),
            }
            for ver in self.pkg.installed_versions
        }

    def set_initial_selector_status(self):
        """
        Sets python selector flag selector_enabled from old selector flag hidePythonApp
        if it is still not set
        """
        if self.Cfg.selector_enabled is not None:
            return

        try:
            old_config = get_ui_config()
            # hidePythonApp:True = disabled selector
            old_selector_status = not old_config['uiSettings']['hidePythonApp']
        except KeyError:
            return
        self.Cfg.selector_enabled = old_selector_status
        self.save()
