# coding=utf-8

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import division
from __future__ import absolute_import
import re

from clselect.baseclselect.pkgmanager import BasePkgManager
from . import (
    scan_python_versions,
    is_major_version,
    CONFIG_DIR,
    create_config_dirs,
    ALT_NAMES,
    ALT_PYTHON_PREFIX)


class PkgManager(BasePkgManager):
    """
    Class responsible for all interactions with Yum, python version
    installation/removal and gathering info about already installed versions
    """

    def _create_config_dirs(self):
        create_config_dirs()

    _config_dir = CONFIG_DIR

    # We should disable all repos but CL's, because Yum will not list anything
    # if some repo is broken

    # TODO We'll improve "Refresh UI button" and implement respective
    # command/option to reset both yum cache and ours in a separate task.
    # {yum_cmd} clean all --disablerepo='*' --enablerepo='cloudlinux-*';

    # https://unix.stackexchange.com/questions/4578/yum-update-error-cant-start-new-thread
    _yum_cmd = 'LANG=C yum --disableplugin=fastestmirror '

    _alt_names = ALT_NAMES

    _redirect_log = '&>/var/log/cl-python-last-yum.log'
    # _redirect_log = ''

    _install_cmd = ''.join([
        _yum_cmd,
        '-y install ',
        _alt_names, '{} ',
        _redirect_log,
    ])

    _remove_cmd = ''.join([
        _yum_cmd,
        '-y remove ',
        _alt_names, '{} ',
        _redirect_log,
    ])

    def _scan_interpreter_versions(self):
        return scan_python_versions()

    def _get_lock_file_path(self, version):
        return ALT_PYTHON_PREFIX + '{}/.lock'.format(
            version.replace('.', ''))

    def checkout_available(self):
        """
        Should return list of major versions available to install from
        currently enabled repos.
            Note, this can be an empty list if no NodeJS version has been
            released to repos yet or in case of network/repos/yum problem.
        OR
            None if our cache is updating right now because it was
            absent/outdated/corrupted/etc.
        :rtype: list | None
        """
        # Data example:
        # alt-python27\n
        # alt-python34\n
        # alt-python36\n
        data = self._read_yum_cache()
        if not data:    # No cache, broken cache, outdated cache etc.
            self.update_yum_cache()
            return None
        available = re.findall('alt-python(\d+)', data)
        available = ['.'.join([version[0], version[1:]]) for version in available]
        return available

    def _verify_action(self, version):
        """Do some common pre-installation/uninstallation checks"""
        if not is_major_version(version):
            return 'Invalid version "{}". ' \
                   'It should be two digits separeted by dot'.format(version)
        working_error = self._check_yum_in_progress()
        if working_error:
            return working_error

    def format_cmd_string_for_installing(self, version):
        """
        Formatting cmd string for install package
        :return: formatted cmd string
        :param version: version of interpreter for installing
        :rtype: str
        """

        return self._install_cmd.format(version.replace('.', ''))

    def format_cmd_string_for_removing(self, version):
        """
        Formatting cmd string for removing package
        :return: formatted cmd string
        :param version: version of interpreter for removing
        :rtype: str
        """

        return self._remove_cmd.format(version.replace('.', ''))

    @staticmethod
    def _is_version_in_use(version):
        # type: (str) -> bool
        """
        Check what passed version isn't used any web-application
        """
        from clselect.clselectpython.apps_manager import ApplicationsManager
        apps_manager = ApplicationsManager()
        return apps_manager.is_version_in_use(version)
