<?php

/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace XrayProfiler;

if (!class_exists('\XrayProfiler\Log')) {
    class Log
    {
        /**
         * @var string
         */
        private $website = '';

        /**
         * @var string
         */
        private $user = '';

        /**
         * @var string
         */
        private $request_uri = '';

        /**
         * @var int
         */
        private $http_code = 0;

        /**
         * @var array<string>
         *
         * PHP Core Exceptions
         */
        public $core_exceptions = array(
            E_ERROR => 'E_ERROR',                         //1
            E_WARNING => 'E_WARNING',                     //2
            E_PARSE => 'E_PARSE',                         //4
            E_NOTICE => 'E_NOTICE',                       //8
            E_CORE_ERROR => 'E_CORE_ERROR',               //16
            E_CORE_WARNING => 'E_CORE_WARNING',           //32
            E_COMPILE_ERROR => 'E_COMPILE_ERROR',         //64
            E_COMPILE_WARNING => 'E_COMPILE_WARNING',     //128
            E_USER_ERROR => 'E_USER_ERROR',               //256
            E_USER_WARNING => 'E_USER_WARNING',           //512
            E_USER_NOTICE => 'E_USER_NOTICE',             //1024
            E_STRICT => 'E_STRICT',                       //2048
            E_RECOVERABLE_ERROR => 'E_RECOVERABLE_ERROR', //4096
            E_DEPRECATED => 'E_DEPRECATED',               //8192
            E_USER_DEPRECATED => 'E_USER_DEPRECATED',     //16384
            E_ALL => 'E_ALL',                             //32767
        );

        /**
         * @var array<array<int|string|null>>
         */
        private $data = array();

        /**
         * @var self|null
         */
        private static $instance = null;

        private function __construct()
        {
        }

        private function __clone()
        {
        }

        /**
         * @return string
         */
        protected function wpHomeConstant()
        {
            if (defined('WP_HOME') && WP_HOME) {
                return (string)WP_HOME;
            }

            return '';
        }

        /**
         * @return string
         */
        protected function wpHomeOption()
        {
            if (function_exists('get_option')) {
                $home = get_option('home');
                if (is_string($home)) {
                    return $home;
                }
            }

            return '';
        }

        /**
         * @return string
         */
        public function website()
        {
            if (! empty($this->website)) {
                return $this->website;
            }

            $wp_home_constant = $this->wpHomeConstant();
            $wp_home_option = $this->wpHomeOption();

            if (! empty($wp_home_constant)) {
                $this->website = $wp_home_constant;
            } elseif (! empty($wp_home_option)) {
                $this->website = $wp_home_option;
            } elseif (is_array($_SERVER) && array_key_exists('SERVER_NAME', $_SERVER)) {
                $this->website = (string)$_SERVER['SERVER_NAME'];
            }

            return $this->website;
        }

        /**
         * @return string
         */
        public function user()
        {
            if (!empty($this->user)) {
                return $this->user;
            }

            $parse = parse_url($this->website());
            if (is_array($parse) and array_key_exists('host', $parse)) {
                $this->user = $parse['host'];
            }

            return $this->user;
        }

        /**
         * @return string
         */
        public function requestUri()
        {
            if (! empty($this->request_uri)) {
                return $this->request_uri;
            }

            if (is_array($_SERVER) && array_key_exists('REQUEST_URI', $_SERVER)) {
                $this->request_uri = (string)$_SERVER['REQUEST_URI'];
            }

            return $this->request_uri;
        }

        /**
         * @return int
         */
        public function httpCode()
        {
            if (! empty($this->http_code)) {
                return $this->http_code;
            }

            if (function_exists('http_response_code')) {
                $this->http_code = (int)http_response_code();
            }

            return $this->http_code;
        }

        /**
         * @return self
         */
        public static function instance()
        {
            if (is_null(self::$instance)) {
                self::$instance = new self();
                self::$instance->clean();
            }

            return self::$instance;
        }

        /**
         * @return array
         */
        public function getData()
        {
            return $this->data;
        }

        /**
         * @param int $errno
         * @param string $errstr
         * @param ?string $errfile
         * @param ?int $errline
         *
         * @return void
         */

        public function setData($errno, $errstr, $errfile = null, $errline = null)
        {
            $this->data[] = array(
                'message' => $errstr,
                'type' => isset($this->core_exceptions[$errno])
                    ? $this->core_exceptions[$errno]
                    : 'Undefined: ' . $errno,
                'filename' => $errfile,
                'lineno' => $errline,
            );
        }

        /**
         * @return string|bool
         */
        public function sendData()
        {
            $data = $this->prepareSentryData($this->data);
            if (empty($data)) {
                return false;
            }
            $sentry_response = $this->sendSentryData($data);
            if (!$sentry_response) {
                return false;
            }
            $this->clean();
            return $sentry_response;
        }

        /**
         * @param array $data
         *
         * @return string|bool
         */
        private function prepareSentryData($data)
        {
            if (empty($data)) {
                return false;
            }
            $formatted = array();
            foreach ($data as $row) {
                $formatted[] = array(
                    "stacktrace" => array(
                        "frames" => array(
                            array(
                                "filename" => $row["filename"],
                                "lineno" => $row["lineno"],
                            )
                        ),
                        "frames_omitted" => null,
                    ),
                    "type" => $row['type'],
                    "value" => $row['message'],
                );
            }

            $xray_version = defined('XRAY_RELEASE') ? XRAY_RELEASE : null;

            $sentry = array(
                "release" => $this->release($xray_version),
                "tags" => array(
                    "php_version" => phpversion(),
                    "xray_version" => $xray_version,
                ),
                "extra" => array(
                    'website' => $this->website(),
                    'request_uri' => $this->requestUri(),
                    'http_code' => $this->httpCode(),
                ),
                "user" => array(
                    "username" => $this->user(),
                ),
                "sentry.interfaces.Exception" => array(
                    "exc_omitted" => null,
                    "values" => $formatted,
                ),
            );

            return json_encode($sentry);
        }

        /**
         * @param string|bool $data
         *
         * @return string|bool
         */
        private function sendSentryData($data)
        {
            if (!function_exists('curl_init') || empty($data)) {
                return false;
            }

            $ch = curl_init();
            $sentry_key = '8e6821f19d214977ace88586bc8569fd';
            $url = 'https://' . $sentry_key . '@cl.sentry.cloudlinux.com/api/23/store/';

            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                'Content-Type: application/json',
                'X-Sentry-Auth: Sentry sentry_version=7,sentry_timestamp=' . time()
                . ',sentry_client=php-curl/1.0,sentry_key=' . $sentry_key,
            ));
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            curl_setopt($ch, CURLOPT_TIMEOUT, 1);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 1);
            curl_setopt($ch, CURLOPT_HEADER, 0);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $response = curl_exec($ch);
            curl_close($ch);

            return $response;
        }

        /**
         * @return void
         */
        public function clean()
        {
            $this->data = array();
            $this->website = '';
            $this->user = '';
            $this->request_uri = '';
            $this->http_code = 0;
        }

        /**
         * Builds the Sentry release string based on X-Ray version.
         *
         * @param  string|null  $xray_version  X-Ray version from XRAY_RELEASE constant.
         *
         * @return string Sentry release string.
         *
         * @since 0.5-23
         */
        public function release($xray_version)
        {
            $version = 'dev';
            if (! empty($xray_version)) {
                preg_match('/\d+\.\d+(\.\d+)?-\d+/', $xray_version, $matches);
                if (! empty($matches)) {
                    $version = $matches[0];
                }
            }

            return 'xray-php-profiler@' . $version;
        }
    }
}
