<?php

/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace XrayProfiler;

if (! class_exists('\XrayProfiler\CollectorBlockingResources')) {
    class CollectorBlockingResources extends Collector
    {
        /**
         * Number of blocking CSS resources found.
         *
         * @var int
         */
        private $css_resource_count = 0;

        /**
         * How many times empty buffer occurred.
         *
         * @var int
         */
        private $empty_buffer_occurred = 0;

        /**
         * @var self|null
         */
        private static $instance = null;

        private function __construct()
        {
            $this->data = [
                'css' => 0,
            ];
        }

        private function __clone()
        {
        }

        /**
         * @return self
         */
        public static function instance()
        {
            if (is_null(self::$instance)) {
                self::$instance = new self();
                self::$instance->clean();
            }

            return self::$instance;
        }

        /**
         * Filter the output buffer contents.
         *
         * @param string $buffer Contents of the output buffer.
         *
         * @return string
         */
        public function startOutputBuffering($buffer)
        {
            if (empty($buffer)) {
                $this->empty_buffer_occurred++;

                return $buffer;
            }

            $this->scanForBlockingCss($buffer);

            return $buffer;
        }

        /**
         * Flush the output buffer.
         *
         * @return void
         */
        public function stopOutputBuffering()
        {
            if (ob_get_contents()) {
                ob_end_flush();
            }
        }

        /**
         * Looks for blocking CSS resources in given HTML code.
         *
         * The number of found resources is added to a local counter.
         *
         * @param string $html
         *
         * @return void
         */
        private function scanForBlockingCss($html)
        {
            $parser = new CssResourceParser();
            $parser->parse($html);

            $this->css_resource_count += $parser->getCount();
        }

        /**
         * @return array
         */
        public function getData()
        {
            $value = $this->css_resource_count;
            if (0 === $value && $this->empty_buffer_occurred > 0) {
                $value = $this->empty_buffer_occurred * -1;
            }

            return [
                'css' => $value,
            ];
        }

        /**
         * {@inheritDoc}
         */
        public function getXrayData()
        {
            return $this->getData();
        }

        /**
         * {@inheritDoc}
         */
        public function clean()
        {
            $this->css_resource_count    = 0;
            $this->empty_buffer_occurred = 0;
            $this->data                  = [];

            return $this;
        }
    }
}
