<?php

/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace XrayProfiler;

if (!class_exists('\XrayProfiler\CollectorCacheability')) {
    class CollectorCacheability extends Collector
    {
        /**
         * AccelerateWP Config instance
         *
         * @var object|false
         */
        private $config = false;

        /**
         * @var array
         */
        private $cookies;

        /**
         * @var array
         */
        private $post;

        /**
         * @var array
         */
        private $get;

        /**
         * @var array
         */
        private $server;

        /**
         * @var self|null
         */
        private static $instance = null;

        /**
         * Constructor.
         *
         */
        private function __construct()
        {
            $this->setConfig();
            $this->setVariables();
        }

        private function __clone()
        {
        }

        /**
         * @return void
         */
        private function setConfig()
        {
            if (
                isset($GLOBALS['rocket_config_path']) &&
                !empty($GLOBALS['rocket_config_path'])
            ) {
                $rocket_config_path = $GLOBALS['rocket_config_path'];
            } elseif (
                defined('WP_ROCKET_CONFIG_PATH') &&
                WP_ROCKET_CONFIG_PATH !== ''
            ) {
                $rocket_config_path = WP_ROCKET_CONFIG_PATH;
            }

            if (
                class_exists('\WP_Rocket\Buffer\Config')
                &&
                isset($rocket_config_path)
            ) {
                $this->config = new \WP_Rocket\Buffer\Config(
                    [
                        'config_dir_path' => $rocket_config_path,
                    ]
                );
            }
        }

        /**
         * @return void
         */
        private function setVariables()
        {
            $this->cookies = !empty($_COOKIE) && is_array($_COOKIE) ? $_COOKIE : array();
            $this->post    = !empty($_POST) && is_array($_POST) ? $_POST : array();
            $this->get     = !empty($_GET) && is_array($_GET) ? $_GET : array();
            $this->server  = !empty($_SERVER) && is_array($_SERVER) ? $_SERVER : array();

            if ($this->post) {
                $this->post = array_intersect_key(
                    // Limit $this->post to the values we need, to save a bit of memory.
                    $this->post,
                    [
                        'wp_customize' => '',
                    ]
                );
            }
        }

        /**
         * @return self
         */
        public static function instance()
        {
            if (is_null(self::$instance)) {
                self::$instance = new self();
                self::$instance->clean();
            }

            return self::$instance;
        }


        /**
         * @return array
         */
        public function getData()
        {
            $this->collectData();
            return $this->data;
        }

        /**
         * Do tests & fill results.
         *
         * @return void
         */
        public function collectData()
        {
            $results = array();

            // Don't process robots.txt && .htaccess files
            // (it has happened sometimes with weird server configuration).
            // Don't process disallowed file extensions (like php, xml, xsl).
            $results['is_disallowed_file'] = $this->isRejectedFile() || $this->isRejectedExtension();

            // Don't cache if in admin.
            $results['is_admin'] = $this->isAdmin();

            // Don't cache if in ajax.
            $results['is_ajax'] = $this->isAjax();

            // Don't process the customizer preview.
            $results['is_preview'] = $this->isCustomizerPreview();

            // Don’t process with query strings parameters,
            // but the processed content is served if the visitor
            // comes from an RSS feed, a Facebook action or Google Adsense tracking.
            $results['is_excluded_by_qs'] = !$this->canProcessQueryString();

            // Don't process these pages.
            $results['is_excluded_by_uri'] = !$this->canProcessUri();

            // Don't process page with rejected cookies.
            // Don't process page when mandatory cookies don't exist.
            $results['is_excluded_by_cookie'] = $this->hasRejectedCookie() || is_array($this->hasMandatoryCookie());

            // Don't process page with these user agents.
            $results['is_excluded_by_ua'] = !$this->canProcessUserAgent();

            // Don't process if mobile detection is activated.
            $results['is_excluded_by_mobile'] = !$this->canProcessMobile();

            // Don't process WordPress search page
            $results['is_search'] = $this->isSearch();

            $this->setData($results);
        }

        /**
         * Tell if the current URI corresponds to a file that must not be processed.
         *
         * @return bool
         */
        public function isRejectedFile()
        {

            $request_uri = $this->getRequestUriBase();

            if (!$request_uri) {
                return false;
            }

            $files = [
                'robots.txt',
                '.htaccess',
            ];

            foreach ($files as $file) {
                if (false !== strpos($request_uri, '/' . $file)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * Tell if the current URI corresponds to a file extension that must not be processed.
         *
         * @return bool
         */
        public function isRejectedExtension()
        {

            $request_uri = $this->getRequestUriBase();

            if (!$request_uri) {
                return false;
            }

            if (strtolower($request_uri) === '/index.php') {
                // `index.php` is allowed.
                return false;
            }

            $extension  = pathinfo($request_uri, PATHINFO_EXTENSION);
            $extensions = [
                'php' => 1,
                'xml' => 1,
                'xsl' => 1,
            ];

            $is_rejected = $extension && isset($extensions[ $extension ]);

            return $is_rejected;
        }

        /**
         * Tell if we're in the admin area (or ajax) or not.
         * Test against ajax added in 2e3c0fa74246aa13b36835f132dfd55b90d4bf9e for whatever reason.
         *
         * @return bool
         */
        public function isAdmin()
        {
            return is_admin();
        }

        /**
         * Tell if we're in the admin area (or ajax) or not.
         * Test against ajax added in 2e3c0fa74246aa13b36835f132dfd55b90d4bf9e for whatever reason.
         *
         * @return bool
         */
        public function isAjax()
        {
            return defined('DOING_AJAX') && DOING_AJAX;
        }

        /**
         * Tell if we're displaying a customizer preview.
         * Test added in 769c7377e764a6a8decb4015a167b34043b4b462 for whatever reason.
         *
         * @return bool
         */
        public function isCustomizerPreview()
        {
            return isset($this->post['wp_customize']);
        }

        /**
         * Don't process with query string parameters, some parameters are allowed though.
         *
         * @return bool
         */
        public function canProcessQueryString()
        {

            $params = $this->getQueryParams();

            if (!$params) {
                return true;
            }

            // The page can be processed if at least one of these parameters is present.
            $allowed_params = [
                'lang'            => 1,
                's'               => 1,
                'permalink_name'  => 1,
                'lp-variation-id' => 1,
            ];

            if (array_intersect_key($params, $allowed_params)) {
                return true;
            }

            // AccelerateWP not installed
            if (!$this->config) {
                return false;
            }

            // The page can be processed if at least one of these parameters is present.
            // @phpstan-ignore-next-line
            $allowed_params = $this->config->get_config('cache_query_strings');

            if (!$allowed_params) {
                // We have query strings but none is in the list set by the user.
                return false;
            }

            $can = (bool) array_intersect_key($params, array_flip($allowed_params));

            return $can;
        }

        /**
         * Some URIs set in the plugin settings must not be processed.
         *
         * @return bool
         */
        public function canProcessUri()
        {

            // AccelerateWP not installed
            if (!$this->config) {
                $uri_pattern = '/(?:.+/)?feed(?:/(?:.+/?)?)?$/|/(?:.+/)?embed/|/(index\.php/)?wp\-json(/.*|$)/';
            } else {
                // URIs not to cache.
                // @phpstan-ignore-next-line
                $uri_pattern = $this->config->get_config('cache_reject_uri');
            }

            if (!$uri_pattern) {
                return true;
            }

            $can = !preg_match('#^(' . $uri_pattern . ')$#i', $this->getRequestUriBase());

            return $can;
        }

        /**
         * Don't process if some cookies are present.
         *
         * @return bool|array
         */
        public function hasRejectedCookie()
        {

            if (!$this->cookies) {
                return false;
            }

            // AccelerateWP not installed
            if (!$this->config) {
                // @phpcs:ignore Generic.Files.LineLength
                $rejected_cookies = '#wordpress_logged_in_.+|wp-postpass_|wptouch_switch_toggle|comment_author_|comment_author_email_#';
            } else {
                // @phpstan-ignore-next-line
                $rejected_cookies = $this->config->get_rejected_cookies();
            }

            if (!$rejected_cookies) {
                return false;
            }

            $excluded_cookies = array();

            foreach (array_keys($this->cookies) as $cookie_name) {
                if (preg_match($rejected_cookies, $cookie_name)) {
                    $excluded_cookies[] = $cookie_name;
                }
            }

            if (!empty($excluded_cookies)) {
                return $excluded_cookies;
            }

            return false;
        }

        /**
         * Don't process if some cookies are NOT present.
         *
         * @return bool|array
         */
        public function hasMandatoryCookie()
        {

            // AccelerateWP not installed
            if (!$this->config) {
                return true;
            }

            // @phpstan-ignore-next-line
            $mandatory_cookies = $this->config->get_mandatory_cookies();

            if (!$mandatory_cookies) {
                return true;
            }

            // @phpstan-ignore-next-line
            $missing_cookies = array_flip(explode('|', $this->config->get_config('cache_mandatory_cookies')));

            if (!$this->cookies) {
                return $missing_cookies;
            }

            foreach (array_keys($this->cookies) as $cookie_name) {
                if (preg_match($mandatory_cookies, $cookie_name)) {
                    unset($missing_cookies[ $cookie_name ]);
                }
            }

            if (empty($missing_cookies)) {
                return true;
            }

            return array_flip($missing_cookies);
        }

        /**
         * Don't process if the user agent is in the forbidden list.
         *
         * @return bool
         */
        public function canProcessUserAgent()
        {

            if (!$this->getServerInput('HTTP_USER_AGENT')) {
                return true;
            }

            // AccelerateWP not installed
            if (!$this->config) {
                $rejected_uas = 'facebookexternalhit|WhatsApp';
            } else {
                // @phpstan-ignore-next-line
                $rejected_uas = $this->config->get_config('cache_reject_ua');
            }

            if (!$rejected_uas) {
                return true;
            }

            $can = !preg_match('#' . $rejected_uas . '#', $this->getServerInput('HTTP_USER_AGENT'));

            return $can;
        }

        /**
         * Don't process if the user agent is in the forbidden list.
         *
         * @return bool
         */
        public function canProcessMobile()
        {

            // AccelerateWP not installed
            if (!$this->config) {
                return true;
            }

            if (!$this->getServerInput('HTTP_USER_AGENT')) {
                return true;
            }

            // @phpstan-ignore-next-line
            if ($this->config->get_config('cache_mobile')) {
                return true;
            }

            // @phpcs:ignore Generic.Files.LineLength
            $uas = '2.0\ MMP|240x320|400X240|AvantGo|BlackBerry|Blazer|Cellphone|Danger|DoCoMo|Elaine/3.0|EudoraWeb|Googlebot-Mobile|hiptop|IEMobile|KYOCERA/WX310K|LG/U990|MIDP-2.|MMEF20|MOT-V|NetFront|Newt|Nintendo\ Wii|Nitro|Nokia|Opera\ Mini|Palm|PlayStation\ Portable|portalmmm|Proxinet|ProxiNet|SHARP-TQ-GX10|SHG-i900|Small|SonyEricsson|Symbian\ OS|SymbianOS|TS21i-10|UP.Browser|UP.Link|webOS|Windows\ CE|WinWAP|YahooSeeker/M1A1-R2D2|iPhone|iPod|Android|BlackBerry9530|LG-TU915\ Obigo|LGE\ VX|webOS|Nokia5800';

            if (preg_match('#^.*(' . $uas . ').*#i', $this->getServerInput('HTTP_USER_AGENT'))) {
                return false;
            }

            // @phpcs:ignore Generic.Files.LineLength
            $uas = 'w3c\ |w3c-|acs-|alav|alca|amoi|audi|avan|benq|bird|blac|blaz|brew|cell|cldc|cmd-|dang|doco|eric|hipt|htc_|inno|ipaq|ipod|jigs|kddi|keji|leno|lg-c|lg-d|lg-g|lge-|lg/u|maui|maxo|midp|mits|mmef|mobi|mot-|moto|mwbp|nec-|newt|noki|palm|pana|pant|phil|play|port|prox|qwap|sage|sams|sany|sch-|sec-|send|seri|sgh-|shar|sie-|siem|smal|smar|sony|sph-|symb|t-mo|teli|tim-|tosh|tsm-|upg1|upsi|vk-v|voda|wap-|wapa|wapi|wapp|wapr|webc|winw|winw|xda\ |xda-';

            if (preg_match('#^(' . $uas . ').*#i', $this->getServerInput('HTTP_USER_AGENT'))) {
                return false;
            }

            return true;
        }

        /**
         * Tell if we're in the WP’s search page.
         *
         * @return bool
         */
        public function isSearch()
        {
            if (!array_key_exists('s', $this->getQueryParams()) || is_admin()) {
                return false;
            }

            /**
             * At this point we’re in the WP’s search page.
             * This filter allows to cache search results.
             *
             * @param bool $cache_search True will force caching search results.
             */
            return !apply_filters('rocket_cache_search', false);
        }

        /**
         * Get the request URI.
         *
         * @return string
         */
        public function getRawRequestUri()
        {
            if ($this->getServerInput('REQUEST_URI') == '') {
                return '';
            }

            return '/' . ltrim($this->getServerInput('REQUEST_URI'), '/');
        }

        /**
         * Get the request URI without the query strings.
         *
         * @return string
         */
        public function getRequestUriBase()
        {
            $request_uri = $this->getRawRequestUri();

            if (!$request_uri) {
                return '';
            }

            $request_uri = explode('?', $request_uri);

            return reset($request_uri);
        }

        /**
         * Get the request method.
         *
         * @return string
         */
        public function getRequestMethod()
        {
            return strtoupper($this->getServerInput('REQUEST_METHOD'));
        }

        /**
         * Get the query string as an array. Parameters are sorted and some are removed.
         *
         * @return array
         */
        public function getQueryParams()
        {
            if (!$this->get) {
                return array();
            }

            if (!$this->config) {
                $config_keys = array(
                    'utm_source' => 0,
                    'utm_medium' => 1,
                    'utm_campaign' => 2,
                    'utm_expid' => 3,
                    'utm_term' => 4,
                    'utm_content' => 5,
                    'mtm_source' => 6,
                    'mtm_medium' => 7,
                    'mtm_campaign' => 8,
                    'mtm_keyword' => 9,
                    'mtm_cid' => 10,
                    'mtm_content' => 11,
                    'pk_source' => 12,
                    'pk_medium' => 13,
                    'pk_campaign' => 14,
                    'pk_keyword' => 15,
                    'pk_cid' => 16,
                    'pk_content' => 17,
                    'fb_action_ids' => 18,
                    'fb_action_types' => 19,
                    'fb_source' => 20,
                    'fbclid' => 21,
                    'campaignid' => 22,
                    'adgroupid' => 23,
                    'adid' => 24,
                    'gclid' => 25,
                    'age-verified' => 26,
                    'ao_noptimize' => 27,
                    'usqp' => 28,
                    'cn-reloaded' => 29,
                    '_ga' => 30,
                    'sscid' => 31,
                    'gclsrc' => 32,
                    '_gl' => 33,
                    'mc_cid' => 34,
                    'mc_eid' => 35,
                    '_bta_tid' => 36,
                    '_bta_c' => 37,
                    'trk_contact' => 38,
                    'trk_msg' => 39,
                    'trk_module' => 40,
                    'trk_sid' => 41,
                    'gdfms' => 42,
                    'gdftrk' => 43,
                    'gdffi' => 44,
                    '_ke' => 45,
                    'redirect_log_mongo_id' => 46,
                    'redirect_mongo_id' => 47,
                    'sb_referer_host' => 48,
                    'mkwid' => 49,
                    'pcrid' => 50,
                    'ef_id' => 51,
                    's_kwcid' => 52,
                    'msclkid' => 53,
                    'dm_i' => 54,
                    'epik' => 55,
                    'pp' => 56,
                    'gbraid' => 57,
                    'wbraid' => 58,
                );
            } else {
                // @phpstan-ignore-next-line
                $config_keys = $this->config->get_config('cache_ignored_parameters');
            }

            // Remove some parameters.
            $params = array_diff_key(
                $this->get,
                $config_keys
            );

            if ($params) {
                ksort($params);
            }

            return $params;
        }

        /**
         * @param string $name
         *
         * @return string
         */
        public function getServerInput($name)
        {
            if (!isset($this->server[$name])) {
                return '';
            }
            return $this->server[$name];
        }

        /**
         * @return array
         */
        public function getXrayData()
        {
            return $this->getData();
        }

        /**
         * @return $this
         */
        public function clean()
        {
            $this->data = array();
            return $this;
        }
    }
}
