<?php

// phpcs:disable PSR1.Files.SideEffects

/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

use XrayProfiler\Log;
use XrayProfiler\CollectorShortcode;
use XrayProfiler\CollectorCacheability;
use XrayProfiler\CollectorBlockingResources;
use XrayProfiler\WebVitalsInjector;

if (defined('ABSPATH') && defined('WPINC') && function_exists('add_filter') && function_exists('json_encode')) {
    if (!defined('XRAY_RELEASE')) {
        define('XRAY_RELEASE', '0.6-34.el7');
    }

    if (!defined('CL_PHP_XRAY_PROFILER_PATH')) {
        define('CL_PHP_XRAY_PROFILER_PATH', '/opt/alt/php-xray/php/profiler');
    }

    try {
        include 'classes/xray-profiler-log.php';

        if (!function_exists('xray_profiler_log')) {
            /**
             * @param int $errno
             * @param string $errstr
             * @param  ?string $errfile
             * @param  ?int $errline
             *
             * @return void
             */
            function xray_profiler_log($errno, $errstr, $errfile = null, $errline = null)
            {
                if (class_exists('\XrayProfiler\Log')) {
                    Log::instance()->setData($errno, $errstr, $errfile, $errline);
                }
            }
        } else {
            return;
        }

        // Send log
        if (!function_exists('xray_profiler_send_log')) {
            /**
             * @return void
             */
            function xray_profiler_send_log()
            {
                if (class_exists('\XrayProfiler\Log')) {
                    Log::instance()->sendData();
                }
            }

            register_shutdown_function('xray_profiler_send_log');
        }
    } catch (Exception $e) {
        return;
    }

    try {
        include 'classes/xray-profiler-collector.php';
        include 'classes/xray-profiler-collector-shortcodes.php';
        include 'classes/xray-profiler-collector-cacheability.php';
        include 'classes/xray-profiler-css-resource-parser.php';
        include 'classes/xray-profiler-collector-blocking-resources.php';
        include 'classes/xray-profiler-web-vitals-injector.php';


        if (!function_exists('xray_profiler_set_error_handler')) {
            /**
             * @return void
             */
            function xray_profiler_set_error_handler()
            {
                set_error_handler(function ($errno, $errstr, $errfile, $errline) {
                    xray_profiler_log($errno, $errstr, $errfile, $errline);
                    return true;
                }, E_ALL);
            }
        }

        if (!function_exists('xray_profiler_restore_error_handler')) {
            /**
             * @return void
             */
            function xray_profiler_restore_error_handler()
            {
                restore_error_handler();
            }
        }

        if (!function_exists('xray_profiler_get_shortcodes_data')) {
            /**
             * @return string
             */
            function xray_profiler_get_shortcodes_data()
            {
                $result = '';

                if (class_exists('\XrayProfiler\CollectorShortcode')) {
                    xray_profiler_set_error_handler();

                    $xray_data = [
                        'shortcodes' => CollectorShortcode::instance()->getXrayData(),
                    ];

                    if (!($json = json_encode($xray_data))) {
                        trigger_error("Can't prepare json for " . __FUNCTION__, 2);
                    } else {
                        $result = $json;
                    }

                    xray_profiler_restore_error_handler();
                }

                return $result;
            }
        }

        // Filters
        if (class_exists('\XrayProfiler\CollectorShortcode')) {
            // @phpcs:ignore PHPCompatibility.Constants.NewConstants.php_int_minFound
            $priority = defined('PHP_INT_MIN') ? PHP_INT_MIN : 0 - PHP_INT_MAX;

            add_filter('pre_do_shortcode_tag', function ($return, $tag, $attr, $m) {
                xray_profiler_set_error_handler();
                $return = CollectorShortcode::instance()->preDoShortcodeTagEarly($return, $tag, $attr, $m);
                xray_profiler_restore_error_handler();
                return $return;
            }, $priority, 4);

            add_filter('pre_do_shortcode_tag', function ($return, $tag, $attr, $m) {
                xray_profiler_set_error_handler();
                $return = CollectorShortcode::instance()->preDoShortcodeTagLate($return, $tag, $attr, $m);
                xray_profiler_restore_error_handler();
                return $return;
            }, PHP_INT_MAX, 4);

            add_filter('do_shortcode_tag', function ($output, $tag, $attr, $m) {
                xray_profiler_set_error_handler();
                $output = CollectorShortcode::instance()->doShortcodeTagLate($output, $tag, $attr, $m);
                xray_profiler_restore_error_handler();
                return $output;
            }, PHP_INT_MAX, 4);
        }

        if (!function_exists('xray_profiler_get_cacheability_data')) {
            /**
             * @return string
             */
            function xray_profiler_get_cacheability_data()
            {
                $result = '';

                if (class_exists('\XrayProfiler\CollectorCacheability')) {
                    xray_profiler_set_error_handler();

                    $xray_data = [
                        'cacheability' => CollectorCacheability::instance()->getXrayData(),
                    ];

                    if (!($json = json_encode($xray_data))) {
                        trigger_error("Can't prepare json for " . __FUNCTION__, 2);
                    } else {
                        $result = $json;
                    }

                    xray_profiler_restore_error_handler();
                }

                return $result;
            }
        }

        if (!function_exists('xray_profiler_get_blocking_resources_data')) {
            /**
             * @return string
             */
            function xray_profiler_get_blocking_resources_data()
            {
                $result = '';

                if (class_exists('\XrayProfiler\CollectorBlockingResources')) {
                    xray_profiler_set_error_handler();

                    $xray_data = [
                        'blocking_resources' => CollectorBlockingResources::instance()->getXrayData(),
                    ];

                    if (!($json = json_encode($xray_data))) {
                        trigger_error("Can't prepare json for " . __FUNCTION__, 2);
                    } else {
                        $result = $json;
                    }

                    xray_profiler_restore_error_handler();
                }

                return $result;
            }
        }

        // Filters
        if (class_exists('\XrayProfiler\CollectorBlockingResources')) {
            // @phpcs:ignore PHPCompatibility.Constants.NewConstants.php_int_minFound
            $priority = defined('PHP_INT_MIN') ? PHP_INT_MIN : 0 - PHP_INT_MAX;
            add_action(
                'template_redirect',
                function () {
                    ob_start(array(
                        CollectorBlockingResources::instance(),
                        'startOutputBuffering'
                    ));
                },
                $priority
            );

            add_action(
                'shutdown',
                array(
                    CollectorBlockingResources::instance(),
                    'stopOutputBuffering'
                ),
                $priority + 1
            );
        }

        // WebVitals Filters
        if (class_exists('\XrayProfiler\WebVitalsInjector')) {
            if (
                @file_exists('/opt/alt/php-xray/php/advanced_metrics.enabled') ||
                (defined('CL_STAGING_MODE') && CL_STAGING_MODE)
            ) {
                add_action(
                    'wp_footer',
                    array(
                        WebVitalsInjector::instance(),
                        'inject'
                    ),
                    1000
                );
            }
        }
    } catch (Exception $e) {
        xray_profiler_log(E_USER_WARNING, $e->getMessage(), $e->getFile(), $e->getLine());
    }
}
