<?php

/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace XrayProfiler;

if (!class_exists('\XrayProfiler\CssResourceParser')) {
    class CssResourceParser
    {
        /**
         * @var int
         */
        private $count = 0;

        /**
         * @param string $html
         *
         * @return void
         */
        public function parse($html)
        {
            $html = $this->removeNoscriptTags($html);
            if (0 === strlen($html)) {
                return;
            }

            if (!preg_match_all('/<link[^>]+href=[\'"](?<href>.+?)[\'"][^>]*>/i', $html, $matches)) {
                return;
            }

            if (!is_array($matches) || empty($matches[0])) {
                return;
            }

            foreach ($matches[0] as $key => $link_fragment) {
                if (preg_match('/rel=["\']preload["\']/i', $link_fragment)) {
                    // skip preloaded links
                    continue;
                }

                if (preg_match('/\bdisabled\b/i', $link_fragment)) {
                    // skip disabled links
                    continue;
                }

                $link_url = $matches['href'][$key];
                $url_path = parse_url($link_url, PHP_URL_PATH);
                if (!is_string($url_path)) {
                    continue;
                }

                $extension = pathinfo($url_path, PATHINFO_EXTENSION);
                if ('css' !== $extension) {
                    continue;
                }

                $this->count++;
            }
        }

        /**
         * @return int
         */
        public function getCount()
        {
            return $this->count;
        }

        /**
         * Removes NOSCRIPT HTML tags from HTML markup.
         *
         * @param string $html HTML code.
         *
         * @return string HTML code without NOSCRIPT tags.
         */
        public function removeNoscriptTags($html)
        {
            return (string) preg_replace('#<noscript(.*?)>(.*?)</noscript>#is', '', $html);
        }
    }
}
